<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\UsersManagementModel;

class UsersManagement extends BaseController
{
    public function index()
    {
        $data = [
            'title' => 'Gestão de Usuários',
            'page' => 'Usuários'
        ];

        // get all users and the name of the restaurant
        $users_model = new UsersManagementModel();
        $users = $users_model->select('users.*, restaurants.name AS restaurant_name')
            ->join('restaurants', 'restaurants.id = users.id_restaurant', 'left')
            ->where('users.id_restaurant', session()->user['id_restaurant'])
            ->findAll();

        $data['users'] = $this->_prepare_users_data($users);
        $data['datatables'] = true;

        return view('dashboard/users_management/index', $data);
    }

    private function _prepare_users_data($users)
    {
        $data = [];

        foreach ($users as $user) {
            $data[] = [
                'id' => $user->id,
                'has_password' => !empty($user->passwrd),
                'username' => $user->username,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'roles' => json_decode($user->roles),
                'blocked_until' => $user->blocked_until,
                'active' => $user->active,
                'last_login' => $user->last_login,
                'created_at' => $user->created_at,
                'updated_at' => $user->updated_at,
                'deleted_at' => $user->deleted_at,
                'restaurant_name' => $user->restaurant_name,
            ];
        }

        return $data;
    }

    // -------------------------------------------------------------------
    // new user
    // -------------------------------------------------------------------
    public function new_user()
    {
        $data = [
            'title' => 'Novo Usuário',
            'page' => 'Usuários'
        ];

        // form validation
        $data['validation_errors'] = session()->getFlashdata('validation_errors');

        // server validation
        $data['server_error'] = session()->getFlashdata('server_error');

        return view('dashboard/users_management/new_user_frm', $data);
    }

    public function new_user_submit()
    {
        // form validation
        $validation = $this->validate($this->_new_user_validation());

        if (!$validation) {
            return redirect()->back()->withInput()->with('validation_errors', $this->validator->getErrors());
        }

        // check if username already exists
        $users_model = new UsersManagementModel();
        $id_restaurant = session()->user['id_restaurant'];
        $username = $this->request->getPost('text_username');
        $email = $this->request->getPost('text_email');

        $results = $users_model->where('id_restaurant', $id_restaurant)
            ->where('username', $username)
            ->orWhere('email', $email)
            ->findAll();
        if (!empty($results)) {
            return redirect()->back()->withInput()->with('server_error', 'Já existe um usuário com o mesmo nome ou email.');
        }

        // insert new user
        $code = bin2hex(random_bytes(16));

        $data = [
            'id_restaurant' => $id_restaurant,
            'username' => $username,
            'name' => $this->request->getPost('text_name'),
            'email' => $email,
            'phone' => $this->request->getPost('text_phone'),
            'roles' => json_encode([$this->request->getPost('select_role')]),
            'active' => 0,
            'code' => $code,
            'created_at' => date('Y-m-d H:i:s')
        ];
        $users_model->insert($data);

        // send e-mail to user to finish registration process by setting a password
        $this->_send_email($data);

        return redirect()->to('/users_management');
    }

    private function _send_email($data)
    {
        // prepare purl - personal url
        $data['purl'] = site_url('/auth/finish_registration/' . $data['code']);

        // config email
        $email = \Config\Services::email();
        $email->setFrom('cigburger@cigburger.com', 'CigBurger');
        $email->setTo($data['email']);
        $email->setSubject('CigBurger - Conclusão de registo de usuário');
        $email->setMailType('html');
        $email->setMessage(view('emails/email_finish_registration', $data));

        // send e-mail and return true or false
        return $email->send();
    }

    // -------------------------------------------------------------------
    // edit user
    // -------------------------------------------------------------------
    public function edit($enc_id)
    {
        // decode id
        $id = Decrypt($enc_id);
        if (!$id) {
            return redirect()->to('/users_management');
        }

        // get user data
        $users_model = new UsersManagementModel();
        $user = $users_model->find($id);

        if (!$user) {
            return redirect()->to('/users_management');
        }

        $data = [
            'title' => 'Usuários',
            'page' => 'Editar usuário',
        ];

        // form validation
        $data['validation_errors'] = session()->getFlashdata('validation_errors');

        // get user role
        $user->roles = json_decode($user->roles)[0];
        $data['user'] = $user;

        // flatpickr
        $data['flatpickr'] = true;

        return view('dashboard/users_management/edit_user_frm', $data);
    }

    public function edit_user_submit()
    {
        // form validation
        $validation = $this->validate($this->_edit_user_validation());
        if(!$validation){
            return redirect()->back()->withInput()->with('validation_errors', $this->validator->getErrors());
        }

        // get user id
        $id = Decrypt($this->request->getPost('hidden_id'));
        if(!$id){
            return redirect()->to('/users_management');
        }

        // get input data
        $role = $this->request->getPost('select_role');
        $active = $this->request->getPost('radio_active');
        $blocked_until = $this->request->getPost('date_blocked_until');

        // prepare blocked until
        if(!empty($blocked_until)){
            $tmp = new \DateTime($blocked_until);
            $tmp->setTime(23,59,59);
            $blocked_until = $tmp->format('Y-m-d H:i:s');
        } else {
            $blocked_until = null;
        }

        // update user
        $users_model = new UsersManagementModel();
        $users_model->update($id, [
            'roles' => json_encode([$role]),
            'active' => $active,
            'blocked_until' => $blocked_until,
            'updated_at' => date('Y-m-d H:i:s')
        ]);

        // redirect to users management
        return redirect()->to('/users_management');
    }

    public function delete_user($enc_id)
    {
        $id = Decrypt($enc_id);
        if(!$id){
            return redirect()->to('/users_management');
        }

        $users_model = new UsersManagementModel();
        $users_model->update($id, [
            'deleted_at' => date('Y-m-d H:i:s')
        ]);

        return redirect()->to('/users_management');
    }

    public function recover_user($enc_id)
    {
        $id = Decrypt($enc_id);
        if(!$id){
            return redirect()->to('/users_management');
        }

        $users_model = new UsersManagementModel();
        $users_model->update($id, [
            'deleted_at' => null
        ]);

        return redirect()->to('/users_management');
    }

    private function _new_user_validation()
    {
        return [
            'text_username' => [
                'label' => 'Usuário',
                'rules' => 'required|min_length[3]|max_length[20]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.',
                ]
            ],
            'text_name' => [
                'label' => 'Nome do usuário',
                'rules' => 'required|min_length[3]|max_length[50]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.',
                ]
            ],
            'text_email' => [
                'label' => 'E-mail',
                'rules' => 'required|valid_email|min_length[5]|max_length[50]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório',
                    'valid_email' => 'O campo {field} deve conter um e-mail válido.',
                    'min_length' => 'O campo {field} deve ter no mínimo {param} caracteres.',
                    'max_length' => 'O campo {field} deve ter no máximo {param} caracteres.',
                ]
            ],
            'text_phone' => [
                'label' => 'Telefone',
                'rules' => 'required|regex_match[/^[9]{1}\d{8}$/]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório',
                    'regex_match' => 'O campo {field} deve conter um número de telefone válido.',
                ]
            ],
            'select_role' => [
                'label' => 'Perfil',
                'rules' => 'required|in_list[admin,user]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório',
                    'in_list' => 'O campo {field} deve conter um valor válido.',
                ]
            ]
        ];
    }

    private function _edit_user_validation()
    {
        return [
            'select_role' => [
                'label' => 'Perfil',
                'rules' => 'required|in_list[admin,user]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'in_list' => 'O campo {field} deve conter um valor válido.'
                ]
            ],
            'radio_active' => [
                'label' => 'Usuário activo',
                'rules' => 'required|in_list[1,0]',
                'errors' => [
                    'required' => 'O campo {field} é obrigatório.',
                    'in_list' => 'O campo {field} deve conter um valor válido.'
                ]
            ],
            'date_blocked_until' => [
                'label' => 'Bloquear usuário até',
                'rules' => 'permit_empty|valid_date[Y-m-d]',
                'errors' => [
                    'valid_date' => 'O campo {field} deve conter uma data válida no formato {param}.'
                ]
            ]
        ];
    }
}
