<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Libraries\ApiResponse;
use App\Models\ApiModel;

class Api extends BaseController
{
    private function _get_project_id()
    {
        $header = $this->request->getHeaderLine('X-API-CREDENTIALS');
        $encrypter = \Config\Services::encrypter();
        $credentials = json_decode($encrypter->decrypt(hex2bin($header)), true);
        return $credentials['project_id'];
    }

    // ============================================================================
    // API ENDPOINTS
    public function api_status()
    {
        $response = new ApiResponse();
        $response->validate_request('get');
        return $response->set_response(200, 'success', [], $this->_get_project_id());
    }

    public function get_restaurant_details()
    {
        $this->response->setContentType('application/json');
        $response = new ApiResponse();
        $response->validate_request('get');

        $api_model = new ApiModel($this->_get_project_id());
        $data = $api_model->get_restaurant_details();
        return $response->set_response(
            200,
            'success',
            $data,
            $this->_get_project_id()
        );
    }

    public function request_checkout()
    {
        $this->response->setContentType('application/json');
        $response = new ApiResponse();
        $response->validate_request('post');

        // responde exactly the same data that was send, just for testing
        $data = $this->request->getJSON(true);

        // analyse request data for data integrity
        $analisys = $this->_analyse_request_data($data);
        if($analisys['status'] == 'error'){
            return $response->set_response(
                400, 
                $analisys['message'], 
                [], 
                $this->_get_project_id()
            );
        }

        // analyse request data for products availability
        $analisys = $this->_analyse_order_products_availability($data);
        if($analisys['status'] == 'error'){
            return $response->set_response(
                400, 
                $analisys['message'], 
                [], 
                $this->_get_project_id()
            );
        }

        // on analysis success
        return $response->set_response(
            200, 
            'success', 
            $data, 
            $this->_get_project_id()
        );
    }

    private function _analyse_request_data($data)
    {
        // check for mandatory data

        // restaurant id
        if(!isset($data['restaurant_id'])){
            return [
                'status' => 'error',
                'message' => 'restaurant_id is mandatory'
            ];
        }

        // check if order contains items collection
        if(!isset($data['order']['items'])){
            return [
                'status' => 'error',
                'message' => 'order.items is mandatory'
            ];
        }

        // check if order contains status
        if(!isset($data['order']['status'])){
            return [
                'status' => 'error',
                'message' => 'order.status is mandatory'
            ];
        }

        // check if order contains items collection
        if($data['order']['status'] != 'paid'){
            return [
                'status' => 'error',
                'message' => 'order.status must be paid'
            ];
        }

        // check if order contains machine_id
        if(!isset($data['machine_id'])){
            return [
                'status' => 'error',
                'message' => 'machine_id is mandatory'
            ];
        }

        // everything is ok
        return [
            'status' => 'success',
            'message' => 'success',
        ];
    }

    private function _analyse_order_products_availability($data)
    {
        // check if all the products are available to satisfy the order
        $api_model = new ApiModel($this->_get_project_id());

        // get order products from api resquest
        $order_products = [];
        foreach($data['order']['items'] as $id => $item){
            $tmp['id_product'] = $id;
            $tmp['quantity'] = $item['quantity'];
            $order_products[] = $tmp;
        }

        // get products from database
        $results = $api_model->get_products_availability($order_products);
        return $results;
    }

    public function request_final_confirmation()
    {
        $this->response->setContentType('application/json');
        $response = new ApiResponse();
        $response->validate_request('post');

        $data = $this->request->getJSON(true);

        // collect data from the request
        $id_restaurant = $data['id_restaurant'];
        $machine_id = $data['machine_id'];
        $total_price = $data['total_price'];
        $order_items = $data['order']['items'];

        $api_model = new ApiModel($this->_get_project_id());
        
        // get last order number from the active restaurant
        $order_number = $api_model->get_last_order_number($id_restaurant);
        
        // increment order number to preparate the next order
        $order_number++;
        
        // add order to database
        $results_order = $api_model->add_order($id_restaurant, $machine_id, $total_price, 'paid', $order_number);

        // on error
        if($results_order['status'] == 'error'){
            return $response->set_response(400, $results_order['message'], [], $this->_get_project_id());
        }

        // get order id
        $id_order = $results_order['id'];

        // add order items to database
        $results_order_items = $api_model->add_order_items($id_order, $order_items);

        // on error
        if($results_order_items['status'] == 'error'){
            return $response->set_response(400, $results_order_items['message'], [], $this->_get_project_id());
        }

        // success
        return $response->set_response(200, 'success', ['id_order' => $id_order, 'order_number' => $order_number], $this->_get_project_id());
    }

    // =====================================================
    // KITCHEN ROUTES
    public function get_pending_orders()
    {
        $this->response->setContentType('application/json');
        $response = new ApiResponse();
        $response->validate_request('get');

        $api_model = new ApiModel($this->_get_project_id());
        $results = $api_model->get_pending_orders();

        // on error
        if($results['status'] == 'error'){
            return $response->set_response(400, $results['message'], [], $this->_get_project_id());
        }

        // success
        return $response->set_response(200, 'success', $results['data'] , $this->_get_project_id());
    }

    public function get_order_details()
    {
        $this->response->setContentType('application/json');
        $response = new ApiResponse();
        $response->validate_request('post');

        $api_model = new ApiModel($this->_get_project_id());

        $data = $this->request->getJSON(true);
        if(empty($data)){
            return $response->set_response(400, 'Invalid parameter', [], $this->_get_project_id());
        }

        $results = $api_model->get_order_details($data['id']);
        if($results['status'] == 'error'){
            return $response->set_response(400, $results['message'], [], $this->_get_project_id());
        }

        return $response->set_response(200, 'success', $results['data'], $this->_get_project_id());
    }

    public function delete_order()
    {
        $this->response->setContentType('application/json');
        $response = new ApiResponse();
        $response->validate_request('post');

        $api_model = new ApiModel($this->_get_project_id());

        $data = $this->request->getJSON(true);
        if(empty($data)){
            return $response->set_response(400, 'Invalid parameter', [], $this->_get_project_id());
        }

        $results = $api_model->delete_order($data['id']);
        if($results['status'] == 'error'){
            return $response->set_response(400, $results['message'], [], $this->_get_project_id());
        }

        return $response->set_response(200, 'success', [], $this->_get_project_id());
    }

    public function get_order_details_with_products()
    {
        $this->response->setContentType('application/json');
        $response = new ApiResponse();
        $response->validate_request('post');

        $api_model = new ApiModel($this->_get_project_id());

        $data = $this->request->getJSON(true);
        if(empty($data)){
            return $response->set_response(400, 'Invalid parameter', [], $this->_get_project_id());
        }

        $results = $api_model->get_order_details_with_products($data['id']);
        if($results['status'] == 'error'){
            return $response->set_response(400, $results['message'], [], $this->_get_project_id());
        }

        return $response->set_response(200, 'success', $results['data'], $this->_get_project_id());
    }

    public function finish_order()
    {
        $this->response->setContentType('application/json');
        $response = new ApiResponse();
        $response->validate_request('post');

        $api_model = new ApiModel($this->_get_project_id());

        $data = $this->request->getJSON(true);
        if(empty($data)){
            return $response->set_response(400, 'Invalid parameter', [], $this->_get_project_id());
        }

        $results = $api_model->finish_order($data['id']);
        if($results['status'] == 'error'){
            return $response->set_response(400, $results['message'], [], $this->_get_project_id());
        }

        return $response->set_response(200, 'success', $results, $this->_get_project_id());
    }
}
